package de.duehl.basics.datetime;

import java.util.List;

/*
 * Copyright 2017 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import de.duehl.basics.collections.CollectionsHelper;
import de.duehl.basics.text.Text;

/**
 * Diese Klasse korrigiert Timestamps.
 *
 * @version 1.01     2017-08-31
 * @author Christian Dühl
 */

public class TimestampCorrector {

    private static final List<String> POSSIBLE_PICTURE_START_STRINGS =
            CollectionsHelper.buildListFrom("England_");

    /** Anzahl Sekunden, die zu den Zeiten der Bilder addiert werden sollen. */
    private final int secondsToAdd;

    /**
     * Konstruktor.
     *
     * @param secondsToAdd
     *            Anzahl Sekunden, die zu den Zeiten der Bilder addiert werden sollen.
     */
    public TimestampCorrector(int secondsToAdd) {
        this.secondsToAdd = secondsToAdd;
    }

    /** Korrigiert den Namen der übergebenen Datei und gibt den korrigierten Namen zurück. */
    public String correct(String filename) {
        String partBeforeTimestamp = determineKnownPartBeforeTimestamp(filename);
        String filenameStartingWithoutTimestamp =
                Text.removeTextAtFrontIfStartsWith(filename, partBeforeTimestamp);
        String correctedFilename = partBeforeTimestamp
                + correctInFilenameStartingWithTimestamp(filenameStartingWithoutTimestamp);
        return correctedFilename;
    }

    private String determineKnownPartBeforeTimestamp(String filename) {
        for (String possibleStart : POSSIBLE_PICTURE_START_STRINGS) {
            if (filename.startsWith(possibleStart)) {
                return possibleStart;
            }
        }

        return "";
    }

    /**
     * Korrigiert den Namen der übergebenen Datei (welche vorn mit dem Timestamp beginnt) und gibt
     * den korrigierten Namen zurück.
     */
    private String correctInFilenameStartingWithTimestamp(String filename) {
        DateAndTime dateAndTime = new DateAndTime(Timestamp.getDateFromTimestamp(filename),
                Timestamp.getTimeFromTimestamp(filename));
        dateAndTime = dateAndTime.addSeconds(secondsToAdd);
        String newTimestamp = dateAndTime.asYyyyMmDdDelimiterHhMmSs("_");
        return Timestamp.switchDateTimePartInName(filename, newTimestamp);
    }

}
