package de.duehl.basics.io.zip;

/*
 * Copyright 2020 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import de.duehl.basics.io.FileHelper;

/**
 * Diese Klasse kann Dateien und Unterverzeichnisse unterhalb eines gemeinsamen Hauptverzeichnisses
 * in ein Zip-Archiv verpacken.
 *
 * @version 1.01     2020-09-17
 * @author Christian Dühl
 */

public class ZipFilesAndDirectoriesBenethOneDirectory {

    /** Das Archiv das angelegt wird. */
    private final ZipFiles zipFiles;

    /**
     * Konstruktor.
     *
     * @param zipFileName
     *            Name der zu erstellenden Zip-Datei.
     * @param directory
     *            Verzeichnis, unterhalb dessen die gepackten Dateien liegen.
     */
    public ZipFilesAndDirectoriesBenethOneDirectory(String zipFileName, String directory) {
        zipFiles = new ZipFiles();
        zipFiles.startAssembling(zipFileName, directory);
    }

    /**
     * Fügt eine Datei zur Sammlung von einzelnen Dateien hinzu.
     *
     * @param fileName
     *            Name der Datei.
     */
    public void addSingleFile(String fileName) {
        zipFiles.addFileToAssembling(fileName);
    }

    /**
     * Fügt alle normalen Dateien in einem Verzeichnis zum Archiv hinzu, ohne seine
     * Unterverzeichnisse zu besuchen.
     *
     * @param directory
     *            Verzeichnis, dessen unmittelbar enthaltenen Dateien hinzugefügt werden sollen.
     */
    public void addAllFilesFromDirectoryWithoutItsSubdirs(String directory) {
        for (String fileName : FileHelper.findAllFilesInMainDirectoryNio2(directory)) {
            addSingleFile(fileName);
        }
    }

    /**
     * Fügt alle normalen Dateien in einem Verzeichnis und allen seinen Unterverzeichnissen erster
     * Ordnung zum Archiv hinzu.
     *
     * @param directory
     *            Verzeichnis, dessen ggf. auch in Unterverzeichnissen enthaltene Dateien
     *            hinzugefügt werden sollen.
     */
    public void addAllFilesFromDirectoryAndItsFirstLevelSubdirs(String directory) {
        addAllFilesFromDirectoryWithoutItsSubdirs(directory);

        for (String subDirectory : FileHelper.findAllSubdirectoriesInMainDirectoryNio2(directory)) {
            addAllFilesFromDirectoryWithoutItsSubdirs(subDirectory);
        }
    }

    /**
     * Fügt alle normalen Dateien in einem Verzeichnis und allen seinen Unterverzeichnissen zum
     * Archiv hinzu.
     *
     * @param directory
     *            Verzeichnis, dessen ggf. auch in Unterverzeichnissen enthaltene Dateien
     *            hinzugefügt werden sollen.
     */
    public void addAllFilesFromDirectoryAndItsSubdirs(String directory) {
        addAllFilesFromDirectoryWithoutItsSubdirs(directory);

        for (String subDirectory : FileHelper.findAllSubdirectoriesInMainDirectoryNio2(directory)) {
            addAllFilesFromDirectoryAndItsSubdirs(subDirectory);
        }
    }

    /**
     * Fügt in einem Verzeichnis aus allen seinen Unterverzeichnissen alle normalen Dateien zum
     * Archiv hinzu, aber keine normalen Dateien auf der höchsten Ebene!
     *
     * @param directory
     *            Verzeichnis, dessen ggf. auch in Unterverzeichnissen enthaltene Dateien
     *            hinzugefügt werden sollen.
     */
    public void addAllFilesFromDirectoryAndItsSubdirsButNoFilesInMainDirectory(String directory) {
        for (String subDirectory : FileHelper.findAllSubdirectoriesInMainDirectoryNio2(directory)) {
            addAllFilesFromDirectoryAndItsSubdirs(subDirectory);
        }
    }

    /** Beendet das Erstellen einer Sammlung von einzelnen Dateien und schreibt die Archivdatei. */
    public void assemblingDone() {
        zipFiles.assemblingDone();
    }

}
