package de.duehl.basics.start;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import de.duehl.basics.datetime.time.TimeHelper;
import de.duehl.basics.io.FileHelper;
import de.duehl.basics.logging.eraser.LogEraser;
import de.duehl.basics.system.SystemTools;
import de.duehl.basics.text.NumberString;
import de.duehl.basics.text.Text;

public class LogEraserTest {

    private static final String TEST_DIR = "c:/temp/log_eraser_test_directory";

    private static final int MAX_TEST_FILE_AGE_IN_DAYS = 40;

    private final LogEraser eraser;

    private final List<String> filenames;
    private final Map<String, Integer> agesByFilename;

    public LogEraserTest() {
        eraser = new LogEraser(TEST_DIR, "", ".txt");
        filenames = new ArrayList<>();
        agesByFilename = new HashMap<>();
        prepareTestDirectory();
    }

    private void prepareTestDirectory() {
        FileHelper.deleteTree(TEST_DIR);
        FileHelper.createDirectory(TEST_DIR);

        for (int ageInDays = 0; ageInDays <= MAX_TEST_FILE_AGE_IN_DAYS; ++ageInDays) {
            String filename = createTestFile(ageInDays);
            filenames.add(filename);
            agesByFilename.put(filename, ageInDays);
        }
    }

    private String createTestFile(int ageInDays) {
        String filename = createTestFilename(ageInDays);
        FileHelper.writeTextToFile("Testdatei", filename);
        setLastModifiedForTestFile(filename, ageInDays);
        return filename;
    }

    private String createTestFilename(int ageInDays) {
        String days = NumberString.addLeadingZeroes(ageInDays, 2);
        String fileBarename = "file_" + days + "_day" + (ageInDays == 1 ? "" : "s") + ".txt";
        String filename = FileHelper.concatPathes(TEST_DIR, fileBarename);
        return filename;
    }

    private void setLastModifiedForTestFile(String filename, int ageInDays) {
        long now = System.currentTimeMillis();
        long dayMillies = ageInDays * TimeHelper.ONE_DAY_MILLIS;
        long lastModified = now - dayMillies;
        FileHelper.setLastModified(filename, lastModified);
    }

    public LogEraserTest setDays(int days) {
        eraser.setDays(days);
        return this;
    }

    public void start() {
        SystemTools.sleepSeconds(1); // Um sicher zu sein, dass der Grenzfall richtig läuft.
        eraser.erase();
        analyseExistingFiles();
    }

    private void analyseExistingFiles() {
        bar();
        int days = eraser.getDays();
        int badCount = 0;
        for (String filename : filenames) {
            int fileDays = agesByFilename.get(filename);
            /*
             * Anmerkung: Da der Eraser erst gestartet wird, nachdem die Dateien Erzeugt wurden,
             * sind die Dateien an der Grenze (etwa 7 Tage alt) ein paar Millisekunden zu alt und
             * werden mit entfernt. Daher wird hier auf
             *     fileDays < days
             * und nicht auf
             *     fileDays <= days
             * geprüft:
             */
            boolean shouldExist = fileDays < days;
            boolean exists = FileHelper.exists(filename);
            boolean ok = shouldExist == exists;
            String existiert = exists ? "existiert" : "existiert nicht";
            String comment = ok ? " Gut." : "SCHLECHT <---- !";
            System.out.println("Die Datei '" + filename + "' " + existiert + "." + comment);
            if (!ok) {
                ++badCount;
            }
        }
        bar();
        if (badCount > 0) {
            System.out.println(badCount + " Fehler!");
        }
        else {
            System.out.println("Alles OK!");
        }
    }

    private void bar() {
        System.out.println(Text.multipleString("-", 80));
    }

    public static void main(String[] args) {
        //new LogEraserTest().start();
        //new LogEraserTest().setDays(7).start();
        new LogEraserTest().setDays(31).start();
    }

}
