package de.duehl.swing.debug.dialog;

/*
 * Copyright 2025 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.BorderLayout;
import java.awt.Color;

import java.util.List;

import javax.swing.JPanel;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.table.TableRowSorter;

import de.duehl.swing.debug.data.DebugMessage;
import de.duehl.swing.debug.dialog.table.DebugMessagesColumnModel;
import de.duehl.swing.debug.dialog.table.DebugMessagesTableModel;
import de.duehl.swing.ui.GuiTools;
import de.duehl.swing.ui.tables.DifferentBackgroundsTableRendererWithChangingSize;

import static de.duehl.swing.debug.dialog.GuiDebugDialog.DEFAULT_FONT_SIZE;

/**
 * Diese Klasse stellt den Inhalt eines Reiters im Dialog mit den Debugmeldungen dar.
 *
 * @version 1.01     2025-07-29
 * @author Christian Dühl
 */

class CategoryPanel {

    private static final Color TABLE_FOREGROUND = new Color(0, 0, 255);
    private static final Color TABLE_BACKGROUND = new Color(240, 240, 255);


    /** Der Panel auf dem die Tabelle zu der Kategorie angezeigt wird. */
    private final JPanel panel;

    /** Die Tabelle. */
    private final JTable table;

    /** Das Modell der anzuzeigenden Tabelle. */
    private final DebugMessagesTableModel tableModel;

    /** Der Renderer der Tabelle. */
    private final DifferentBackgroundsTableRendererWithChangingSize tableRenderer;

    /** Gibt an, ob in der Tabelle auch die Kategorie angezeigt werden soll. */
    private boolean showCategory;

    /**
     * Konstruktor.
     * @param messagesByCategory
     *            Die eingegangenen Meldungen nach Kategorie.
     */
    public CategoryPanel(List<DebugMessage> messages) {
        panel = new JPanel();

        table = new JTable();
        tableModel = new DebugMessagesTableModel(messages);
        tableRenderer = new DifferentBackgroundsTableRendererWithChangingSize(TABLE_FOREGROUND,
                TABLE_BACKGROUND);

        showCategory = false;

        init();
    }

    private void init() {
        panel.setLayout(new BorderLayout());
    }

    /** Legt fest, dass in der Tabelle auch die Kategorie angezeigt werden soll. */
    public void showCategory() {
        showCategory = true;
        tableModel.showCategory();
    }

    /** Erstellt die Oberfläche des Panels. */
    public void crateGui() {
        initTable();

        panel.add(GuiTools.createScrollPane(table), BorderLayout.CENTER);
    }

    private void initTable() {
        initTableModel();
        initTableColumnModel();
        initTableRenderer();
        setTableSelectionMode();
        setTableRowHight();
        switchReorderingOfTableColumnsOff();
        initTableSorter();
    }

    private void initTableModel() {
        table.setModel(tableModel);
    }

    private void initTableColumnModel() {
        table.setColumnModel(new DebugMessagesColumnModel(showCategory));
    }

    private void initTableRenderer() {
        tableRenderer.setFontSize(DEFAULT_FONT_SIZE);
        table.setDefaultRenderer(Object.class, tableRenderer);
    }

    private void setTableSelectionMode() {
        table.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
    }

    private void setTableRowHight() {
        tableRenderer.initRowHeightOfTable(table);
    }

    private void switchReorderingOfTableColumnsOff() {
        table.getTableHeader().setReorderingAllowed(false); // verschieben der Spalten verhindern
    }

    private void initTableSorter() {
        TableRowSorter<DebugMessagesTableModel> sorter = new TableRowSorter<>(tableModel);
        table.setRowSorter(sorter);
    }

    public void increaseFontSize() {
        tableRenderer.increaseFontSize();
        tableRenderer.initRowHeightOfTable(table);
        table.repaint();
    }

    public void standardFontSize() {
        tableRenderer.setFontSize(DEFAULT_FONT_SIZE);
        tableRenderer.initRowHeightOfTable(table);
        table.repaint();
    }

    public void decreaseFontSize() {
        tableRenderer.decreaseFontSize();
        tableRenderer.initRowHeightOfTable(table);
        table.repaint();
    }

    /** Getter für den Panel auf dem die Tabelle zu der Kategorie angezeigt wird. */
    public JPanel getPanel() {
        return panel;
    }

}
