package de.duehl.swing.ui.components.elements;

/*
 * Copyright 2021 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.BorderLayout;
import java.awt.CardLayout;
import java.awt.Component;
import java.util.ArrayList;
import java.util.List;

import javax.swing.JPanel;

import de.duehl.swing.ui.GuiTools;
import de.duehl.swing.ui.buttons.creator.ButtonCreator;
import de.duehl.swing.ui.buttons.creator.StandardButtonCreator;
import de.duehl.swing.ui.layout.VerticalLayout;

/**
 * Diese Klasse stellt einen vertikalen Tabulator dar, also einen Anzeigebereich mit austauschbaren
 * Inhalten und einen Bereich mit Buttons, um diese zu wechseln.
 *
 * Bekommt übergeben:
 *     addTab(String, JComponent) ... davon mehrere
 * Stellt die Tabs links dar untereinander (als Buttons oder geschickt formatierte Label)
 * wechselt rechts mittels Card-Layout die Inhalte!
 *
 * @version 1.01     2021-05-07
 * @author Christian Dühl
 */

public class VerticalTabulator {

    /** Der Panel mit den Buttons zu den verschiedenen Reitern. */
    private final JPanel buttonPanel;

    /** Der Panel der die wechselnden Inhalte der Reiter anzeigen kann. */
    private final JPanel contentPanel;

    /** Das Layout für den Panel mit den Karten. */
    private final CardLayout cardLayout;

    /** Liste mit den Beschriftungen der Buttons. */
    private final List<String> buttonTexts;

    /** Objekt das die Buttons erstellt. */
    private ButtonCreator buttonCreator;

    /** Konstruktor. */
    public VerticalTabulator() {
        buttonPanel = new JPanel();
        contentPanel = new JPanel();

        buttonPanel.setLayout(new VerticalLayout(5, VerticalLayout.BOTH));

        cardLayout = new CardLayout();
        contentPanel.setLayout(cardLayout);

        buttonTexts = new ArrayList<>();
        buttonCreator = new StandardButtonCreator();
    }

    /**
     * Setzt das Objekt das die Buttons erstellt.
     *
     * Default ist der StandardButtonCreator.
     */
    public void setButtonCreator(ButtonCreator buttonCreator) {
        this.buttonCreator = buttonCreator;
    }

    /**
     * Fügt einen Tabulator hinzu.
     *
     * @param buttonText
     *            Beschriftung des Buttons.
     * @param content
     *            Anzuzeigender Inhalt.
     */
    public void addTabulator(String buttonText, Component content) {
        if (buttonTexts.contains(buttonText)) {
            throw new IllegalArgumentException(
                    "Doppelte Button-Beschriftung: '" + buttonText + "'!");
        }
        buttonTexts.add(buttonText);
        contentPanel.add(content, buttonText);
        buttonPanel.add(createButton(buttonText));
    }

    private Component createButton(String buttonText) {
        return buttonCreator.createButton(buttonText, () -> switchToTabulator(buttonText));
    }

    private void switchToTabulator(String buttonText) {
        cardLayout.show(contentPanel, buttonText);
    }

    /** Erzeugt einen Panel mit den Buttons der Reiter auf der linken Seite. */
    public Component createPanelWithButtonsOnTheLeft() {
        showFirstTabulator();

        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());

        panel.add(buttonPanel, BorderLayout.WEST);
        panel.add(contentPanel, BorderLayout.CENTER);

        return panel;
    }

    private void showFirstTabulator() {
        if (buttonTexts.isEmpty()) {
            throw new RuntimeException("Es wurden vergessen, mit addTabluator() Tabulatoren zum "
                    + "VerticalTabulator hinzuzufügen!");
        }
        switchToTabulator(buttonTexts.get(0));
    }

    public void createBorders() {
        createButtonsBorder();
        createContentBorder();
    }

    public void createButtonsBorder() {
        GuiTools.createTitle(buttonPanel);
    }

    public void createContentBorder() {
        GuiTools.createTitle(contentPanel);
    }

}
