package de.duehl.swing.ui.dialogs;

/*
 * Copyright 2019 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Image;
import java.awt.Point;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import javax.swing.JButton;
import javax.swing.JPanel;

import de.duehl.basics.io.Charset;
import de.duehl.basics.io.FileHelper;
import de.duehl.swing.ui.GuiTools;
import de.duehl.swing.ui.components.MultipleElementsPanel;
import de.duehl.swing.ui.dialogs.base.ModalDialogBase;

/**
 * Diese Klasse öffnet einen Dialog zur Auswahl des Charsets.
 *
 * @version 1.02     2019-05-24
 * @author Christian Dühl
 */

public class CharsetDialog extends ModalDialogBase {

    private static final Color CHARSET_BUTTON_BACKGROUND = new Color(230, 230, 255);
    private static final Color DEFAULT_CHARSET_BUTTON_BACKGROUND = new Color(230, 255, 230);


    /** Das vom Benutzer ausgewählte Charset (oder bei Abbruch das Default-Charset). */
    private Charset selectedCharset;

    /** Der Button des Default-Charsets, um diesem den Focus zu geben. */
    private JButton defaultCharsetButton;

    /**
     * Konstruktor.
     *
     * @param parentLocation
     *            Position des Rahmens der Oberfläche, vor der dieser Dialog
     *            erzeugt wird.
     * @param programImage
     *            Image für den Dialog.
     * @param filename
     *            Name der Datei für den Titel des Dialogs.
     */
    public CharsetDialog(Point parentLocation, Image programImage, String filename) {
        super(parentLocation, programImage, createTitle(filename));
        selectedCharset = Charset.getDefaultCharset();

        fillDialog();
        defaultCharsetButton.requestFocusInWindow();
    }

    private static String createTitle(String filename) {
        return "Bitte das Charset für die Datei" + FileHelper.getBareName(filename) + " auswählen.";
    }

    /** Baut die Gui auf. */
    @Override
    protected void populateDialog() {
        add(createCharsetButtonPart(), BorderLayout.CENTER);
        add(createButtonPart(), BorderLayout.SOUTH);

        addEscapeBehaviour();
    }

    private Component createCharsetButtonPart() {
        List<JButton> charsetButtons = createCharsetButtons();
        return createMultiplePanel(charsetButtons);
    }

    private List<JButton> createCharsetButtons() {
        List<JButton> charsetButtons = new ArrayList<>();

        for (Charset charset : sortedCharsets()) {
            charsetButtons.add(createCharsetButtonWithSelection(charset));
        }

        return charsetButtons;
    }

    protected List<Charset> sortedCharsets() {
        List<Charset> sortedCharsets = new ArrayList<>();

        for (Charset charset : Charset.values()) {
            sortedCharsets.add(charset);
        }

        Collections.sort(sortedCharsets, new Comparator<Charset>() {
            @Override
            public int compare(Charset c1, Charset c2) {
                String n1 = c1.getCharsetAsString();
                String n2 = c2.getCharsetAsString();
                return n1.compareTo(n2);
            }
        });

        return sortedCharsets;
    }

    private JButton createCharsetButtonWithSelection(Charset charset) {
        JButton button = createCharsetButton(charset);
        if (Charset.getDefaultCharset() == charset) {
            //button.setBorder(BorderFactory.createEmptyBorder(2, 2, 2, 2));
            button.setBackground(DEFAULT_CHARSET_BUTTON_BACKGROUND);
            //button.setForeground(DEFAULT_CHARSET_BUTTON_BACKGROUND);
            defaultCharsetButton = button;
        }
        return button;
    }

    private JButton createCharsetButton(Charset charset) {
        //JButton button = new JButton("     " + charset.getCharsetAsString() + "     ");
        JButton button = new JButton(charset.getCharsetAsString());
        button.addActionListener(e -> buttonPressed(charset));
        button.setOpaque(true);
        button.setBackground(CHARSET_BUTTON_BACKGROUND);
        //button.setForeground(CHARSET_BUTTON_BACKGROUND);
        GuiTools.biggerFont(button, 5);
        return button;
    }

    private void buttonPressed(Charset charset) {
        selectedCharset  = charset;
        closeDialog();
    }

    private Component createMultiplePanel(List<JButton> charsetButtons) {
        int maximalNumberOfColumns = 3; // hat Vorrang
        int maximalNumberOfElementsPerColumn = 3;

        return new MultipleElementsPanel<JButton>(charsetButtons, maximalNumberOfColumns,
                maximalNumberOfElementsPerColumn);
    }

    /** Erzeugt den unteren Bereich mit den Buttons. */
    private Component createButtonPart() {
        JPanel panel = new JPanel();
        setColors(panel);
        panel.setLayout(new BorderLayout());

        panel.add(createQuitButton(), BorderLayout.EAST);

        return panel;
    }

    private JButton createQuitButton() {
        JButton button = new JButton("Schließen");
        setColors(button);
        button.addActionListener(e -> closeDialog());
        button.setFocusable(false);

        return button;
    }

    /** Getter für das vom Benutzer ausgewählte Charset (oder bei Abbruch das Default-Charset). */
    public Charset getSelectedCharset() {
        return selectedCharset;
    }

}
