package de.duehl.swing.ui.menu.collection.tools;

/*
 * Copyright 2020 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import de.duehl.basics.collections.CollectionsHelper;
import de.duehl.basics.text.html.generation.SwingHtmlBuilder;
import de.duehl.swing.ui.menu.collection.StoredMyMenuItem;

/**
 * Diese Klasse erzeugt eine HTML-Beschreibung der übergebenen Menüeinträge.
 *
 * @version 1.01     2020-12-09
 * @author Christian Dühl
 */

class MyMenuItemsHtmlDescriptionGenerator {

    /** Liste der gespeicherten Menüeinträge. */
    private final List<StoredMyMenuItem> collection;

    /** Die hier erzeugte HTML-Ausgabe als Builder. */
    private final SwingHtmlBuilder html;

    /** Eventuell gesetzter CSS-Code. */
    private String css;

    /**
     * Gibt an, ob die Belegungen nach den tasten sortiert werden sollen. Anderenfalls sind sie in
     * der Reihenfolge, in der sie in den Menüs vorkommen.
     */
    private boolean sortByKeys;

    /**
     * Konstruktor.
     *
     * @param collection
     *            Liste der gespeicherten Menüeinträge.
     */
    public MyMenuItemsHtmlDescriptionGenerator(List<StoredMyMenuItem> collection) {
        this.collection = CollectionsHelper.copyList(collection); // Da im Zweifelsfall sortiert
                                                                  // wird, wollen wir die originale
                                                                  // Liste nicht verändern!
        html = new SwingHtmlBuilder();
        sortByKeys = false;
    }

    /** Legt den CSS-Code fest. */
    public void setCss(String css) {
        this.css = css;
    }

    /** Sortiert die Belegung nach den zu drückenden Tasten. */
    public void sortByKeys() {
        sortByKeys = true;
    }

    /** Erzeugt die Beschreibung. */
    public void buildDescription() {
        sortIfNecessary();
        appendHtmlHead();
        appendTableHead();
        appendTableBody();
        appendTableFoot();
        appendHtmlFoot();
    }

    private void sortIfNecessary() {
        if (sortByKeys) {
            sortCollectionByKeys();
        }
    }

    private void sortCollectionByKeys() {
        Collections.sort(collection, new Comparator<StoredMyMenuItem>() {
            @Override
            public int compare(StoredMyMenuItem m1, StoredMyMenuItem m2) {
                {
                    int modifiers1 = m1.getModifiers();
                    int modifiers2 = m2.getModifiers();
                    if (modifiers1 != modifiers2) {
                        return modifiers1 - modifiers2;
                    }
                }

                {
                    int keyCode1 = m1.getKeyCode();
                    int keyCode2 = m2.getKeyCode();
                    return keyCode1 - keyCode2;
                }
            }
        });
    }

    private void appendHtmlHead() {
        String pageTitle = "Übersicht über die Tastaturbefehle";
        if (sortByKeys) {
            pageTitle += " in Reihenfolge der Tasten";
        }
        if (css.isEmpty()) {
            html.appendHtml5HeadWithOwnCss(pageTitle);
        }
        else {
            html.appendHtml5HeadWithOwnExtendedCss(pageTitle, css);
        }

        String header = "Übersicht über die im Menü hinterlegten Tastaturbefehle";
        if (sortByKeys) {
            header += " in Reihenfolge der Tasten";
        }
        html.appendH1(header);
    }

    private void appendTableHead() {
        html.appendOpeningTableWithBorderWidth(2);

        html.appendOpeningTr();
        html.appendLeftAlignedTh("Tastaturbefehl");
        html.appendLeftAlignedTh("Funktion");
        html.appendClosingTr();
    }

    private void appendTableBody() {
        for (StoredMyMenuItem storedMyMenuItem : collection) {
            if (storedMyMenuItem.isKeyCodeAndModifiersStored()) {
                appendTableRow(storedMyMenuItem);
            }
        }
    }

    private void appendTableRow(StoredMyMenuItem storedMyMenuItem) {
        html.appendOpeningTr();
        html.appendTd(
                MyMenuItemCollectionHelper.generateKeyboardCommandDescription(storedMyMenuItem));
        html.appendTd(storedMyMenuItem.getText());
        html.appendClosingTr();
    }

    private void appendTableFoot() {
        html.appendClosingTable();
    }

    private void appendHtmlFoot() {
        html.appendFoot();
    }

    public String getHtmlDescription() {
        return html.toString();
    }

}
