package de.duehl.swing.ui.colors;

/*
 * Copyright 2024 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import static org.junit.Assert.*;

import java.awt.Color;
import java.util.List;

import org.junit.Test;

import de.duehl.basics.collections.CollectionsHelper;

public class ColorToolTest {

    @Test
    public void create() {
        ColorTool tool = new ColorTool();
        assertNotNull(tool);
    }

    @Test
    public void javaColorToHexColor1() {
        Color color = Color.RED;
        String expected = "FF0000";
        String actual = ColorTool.javaColorToHexColor(color);
        assertEquals(expected, actual);
    }

    @Test
    public void javaColorToHexColor2() {
        Color color = Color.WHITE;
        String expected = "FFFFFF";
        String actual = ColorTool.javaColorToHexColor(color);
        assertEquals(expected, actual);
    }

    @Test
    public void javaColorToHexColor3() {
        Color color = Color.BLACK;
        String expected = "000000";
        String actual = ColorTool.javaColorToHexColor(color);
        assertEquals(expected, actual);
    }

    @Test
    public void hexColorToJavaColor1() {
        String color = "FF0000";
        Color expected = Color.RED;
        Color actual = ColorTool.hexColorToJavaColor(color);
        assertEquals(expected, actual);
    }

    @Test
    public void hexColorToJavaColor2() {
        String color = "FFFFFF";
        Color expected = Color.WHITE;
        Color actual = ColorTool.hexColorToJavaColor(color);
        assertEquals(expected, actual);
    }

    @Test
    public void hexColorToJavaColor3() {
        String color = "000000";
        Color expected = Color.BLACK;
        Color actual = ColorTool.hexColorToJavaColor(color);
        assertEquals(expected, actual);
    }

    @Test
    public void hexColorListToJavaColorList() {
        List<String> hexColors = CollectionsHelper.buildListFrom("FF0000", "FFFFFF", "000000");
        List<Color> actualColors = ColorTool.hexColorListToJavaColorList(hexColors);
        List<Color> expectedColors = CollectionsHelper.buildListFrom(
                Color.RED, Color.WHITE, Color.BLACK);
        assertEquals(expectedColors, actualColors);
    }

    @Test
    public void rgbToHexColorWhite() {
        int red = 255;
        int green = 255;
        int blue = 255;
        String expected = "ffffff";
        String actual = ColorTool.rgbToHexColor(red, green, blue);
        assertEquals(expected, actual);
    }

    @Test
    public void rgbToHexColorBlack() {
        int red = 0;
        int green = 0;
        int blue = 0;
        String expected = "000000";
        String actual = ColorTool.rgbToHexColor(red, green, blue);
        assertEquals(expected, actual);
    }

    @Test (expected = IllegalArgumentException.class)
    public void rgbToHexColorTooSmall() {
        int red = -4;
        int green = 255;
        int blue = 255;
        ColorTool.rgbToHexColor(red, green, blue);
    }

    @Test (expected = IllegalArgumentException.class)
    public void rgbToHexColorTooBih() {
        int red = 0;
        int green = 256;
        int blue = 255;
        ColorTool.rgbToHexColor(red, green, blue);
    }

    @Test
    public void rgbToHexColorAutomaticColor() {
        int red = 200;
        int green = 230;
        int blue = 255;
        String expected = "c8e6ff";
        String actual = ColorTool.rgbToHexColor(red, green, blue);
        assertEquals(expected, actual);
    }

    @Test
    public void rgbToHexColorAutomaticColorOk() {
        int red = 200;
        int green = 255;
        int blue = 230;
        String expected = "c8ffe6";
        String actual = ColorTool.rgbToHexColor(red, green, blue);
        assertEquals(expected, actual);
    }

    @Test
    public void rgbToHexColorAutomaticChangedColor() {
        int red = 220;
        int green = 255;
        int blue = 240;
        String expected = "dcfff0";
        String actual = ColorTool.rgbToHexColor(red, green, blue);
        assertEquals(expected, actual);
    }

    @Test
    public void rgbToHexColorEditableFieldColor() {
        int red = 255;
        int green = 255;
        int blue = 180;
        String expected = "ffffb4";
        String actual = ColorTool.rgbToHexColor(red, green, blue);
        assertEquals(expected, actual);
    }

    @Test
    public void rgbToHexColorNotEditableFieldColor() {
        int red = 240;
        int green = 240;
        int blue = 240;
        String expected = "f0f0f0";
        String actual = ColorTool.rgbToHexColor(red, green, blue);
        assertEquals(expected, actual);
    }

    @Test
    public void rgbToHexColorAutomaticFailureColor() {
        int red = 255;
        int green = 255;
        int blue = 180;
        String expected = "ffffb4";
        String actual = ColorTool.rgbToHexColor(red, green, blue);
        assertEquals(expected, actual);
    }

    @Test
    public void rgbToHexColorReachThroughColor() {
        int red = 200;
        int green = 255;
        int blue = 230;
        String expected = "c8ffe6";
        String actual = ColorTool.rgbToHexColor(red, green, blue);
        assertEquals(expected, actual);
    }

    @Test
    public void rgbToHexColorReachThroughChangedColor() {
        int red = 255;
        int green = 190;
        int blue = 190;
        String expected = "ffbebe";
        String actual = ColorTool.rgbToHexColor(red, green, blue);
        assertEquals(expected, actual);
    }

    @Test
    public void rgbToHexColorDummyPersonsColor() {
        int red = 255;
        int green = 230;
        int blue = 200;
        String expected = "ffe6c8";
        String actual = ColorTool.rgbToHexColor(red, green, blue);
        assertEquals(expected, actual);
    }

}
