package de.duehl.vocabulary.japanese.logic.test;

import java.util.List;

import de.duehl.vocabulary.japanese.common.data.TranslationDirection;
import de.duehl.vocabulary.japanese.common.persistence.Options;
import de.duehl.vocabulary.japanese.data.Vocable;
import de.duehl.vocabulary.japanese.tools.VocabularyTools;

/**
 * Diese Klasse prüft, ob der Benutzer richtig übersetzt hat.
 *
 * @version 1.01     2024-04-24
 * @author Christian Dühl
 */

public class UserAnswerCorrectnessChecker {

    /** Die Programmoptionen. */
    private final Options options;

    /** Die vom Benutzer eingegebenen Übersetzung zum Vergleich. */
    private final String compareTranslationByUser;

    /**
     * Die Liste der zu den dem Benutzer angezeigten Informationen (bei Japanisch in Deutsch sind
     * das kana und kanji, bei Deutsch in Japanisch ist das germanTerm) passenden Vokabeln.
     */
    private final List<Vocable> matchingVocables;

    /** Gibt an, ob der Benutzer richtig übersetzt hat. */
    private boolean correct;

    /** Falls der Benutzer richtig übersetzt hat, ist dies die zugehörige Vokabel. */
    private Vocable correctVocable;

    /**
     * Konstruktor.
     *
     * @param options
     *            Die Programmoptionen.
     * @param translationByUser
     *            Die vom Benutzer eingegebenen Übersetzung.
     * @param matchingVocables
     *            Die Liste der zu den dem Benutzer angezeigten Informationen (bei Japanisch in
     *            Deutsch sind das kana und kanji, bei Deutsch in Japanisch ist das germanTerm)
     *            passenden Vokabeln.
     */
    public UserAnswerCorrectnessChecker(Options options,
            String translationByUser, List<Vocable> matchingVocables) {
        this.options = options;
        this.matchingVocables = matchingVocables;

        compareTranslationByUser = VocabularyTools.createCompareTranslation(translationByUser);
        correct = false;
    }

    /** Führt die Prüfung durch. */
    public void check() {
        TranslationDirection translationDirection = options.getTranslationDirection();

        if (translationDirection == TranslationDirection.JAPANESE_TO_GERMAN) {
            checkJapaneseToGerman();
        }
        else if (translationDirection == TranslationDirection.GERMAN_TO_JAPANESE) {
            checkGermanToJapanese();
        }
        else {
            throw new RuntimeException("Unbekannte Übersetzungsrichtung " + translationDirection);
        }
    }

    private void checkJapaneseToGerman() {
        correct = false;

        for (Vocable vocable : matchingVocables) {
            if (vocable.getCompareTranslations().contains(compareTranslationByUser)) {
                correct = true;
                correctVocable = vocable;
                break;
            }
        }
    }

    private void checkGermanToJapanese() {
        correct = false;

        for (Vocable vocable : matchingVocables) {
            if (vocable.getCompareKana().equals(compareTranslationByUser)
                    || vocable.getCompareKanji().equals(compareTranslationByUser)
                    || vocable.getCompareRomaji().equals(compareTranslationByUser)) {
                correct = true;
                correctVocable = vocable;
                break;
            }
        }
    }

    /** Gibt an, ob der Benutzer richtig übersetzt hat. */
    public boolean isCorrect() {
        return correct;
    }

    /** Falls der Benutzer richtig übersetzt hat, wird hier die zugehörige Vokabel zurückgegeben. */
    public Vocable getCorrectVocable() {
        return correctVocable;
    }

}
