package de.duehl.vocabulary.japanese.ui.dialog.kanji.display;

import java.awt.BorderLayout;
import java.awt.Component;

import javax.swing.JPanel;

import de.duehl.basics.text.Text;
import de.duehl.swing.ui.components.selections.TextAreaSelection;
import de.duehl.swing.ui.components.selections.tools.SelectionsHelper;
import de.duehl.swing.ui.layout.VerticalLayout;
import de.duehl.vocabulary.japanese.data.symbol.Kanji;

/**
 * Diese Klasse stellt die deutschen Bedeutungen und die Lesungen eines einzelnen Kanji auf einem
 * Panel dar.
 *
 * Gezeigt werden die deutschen Bedeutungen, die ON-Lesungen und die kun-Lesungen.
 *
 * @version 1.01     2026-01-01
 * @author Christian Dühl
 */

public class KanjiGermanMeaningAndLesungenPanel {

    /** Das angezeigte Kanji-Zeichen. */
    private final Kanji kanji;

    /** Der Panel auf dem das Kanji dargestellt wird. */
    private final JPanel panel;

    /* Gui-Elemente: */
    private final TextAreaSelection germanMeaningSelection;
    private final TextAreaSelection onLesungenSelection;
    private final TextAreaSelection kunLesungenSelection;

    /**
     * Konstruktor.
     *
     * @param kanji
     *            Das angezeigte Kanji-Zeichen.
     */
    public KanjiGermanMeaningAndLesungenPanel(Kanji kanji) {
        this.kanji = kanji;

        panel = new JPanel();

        germanMeaningSelection = new TextAreaSelection("Deutsche Bedeutung");
        onLesungenSelection = new TextAreaSelection("ON-Lesungen");
        kunLesungenSelection = new TextAreaSelection("kun-Lesungen");

        init();
        createGui();
        showKanji();
    }

    private void init() {
        SelectionsHelper.initSelectionAsFocusableViewer(germanMeaningSelection);
        SelectionsHelper.initSelectionAsFocusableViewer(onLesungenSelection);
        SelectionsHelper.initSelectionAsFocusableViewer(kunLesungenSelection);
    }

    private void createGui() {
        panel.setLayout(new BorderLayout());
        //GuiTools.createTitle(panel); Das sollte man wohl außerhalb entscheiden.

        panel.add(createGermanMeaningOnLesungenAndKunLesungenPart(),
                BorderLayout.CENTER);
    }

    private Component createGermanMeaningOnLesungenAndKunLesungenPart() {
        JPanel panel = new JPanel();
        panel.setLayout(new VerticalLayout(0, VerticalLayout.BOTH));

        panel.add(germanMeaningSelection.getPanel());
        panel.add(onLesungenSelection.getPanel());
        panel.add(kunLesungenSelection.getPanel());

        return panel;
    }

    /** Zeigt das Kanji an. */
    private void showKanji() {
        germanMeaningSelection.setText(Text.joinWithLineBreak(kanji.getGermanMeanings()));
        onLesungenSelection.setText(Text.joinWithLineBreak(kanji.getOnLesungen()));
        kunLesungenSelection.setText(Text.joinWithLineBreak(kanji.getKunLesungen()));
    }

    /**
     * Getter für den Panel auf dem die deutschen Bedeutungen und die Lesungen des Kanji
     * dargestellt werden.
     */
    public Component getPanel() {
        return panel;
    }

}
