package de.duehl.vocabulary.japanese.website.update.ownlists.interest;

import static de.duehl.vocabulary.japanese.common.persistence.SessionManager.VOCABLE_TRAINER_DIRECTORY;

import java.util.ArrayList;
import java.util.List;

import de.duehl.basics.io.FileHelper;
import de.duehl.basics.text.handling.BooleanHandling;
import de.duehl.vocabulary.japanese.website.update.ownlists.data.OwnListInterestAndVersionEntry;
import de.duehl.vocabulary.japanese.website.update.ownlists.parser.OwnListVersionAndInterestParser;

/**
 * Diese Klasse hält den Pfad für die Datei
 *     Eigene_Listen_Interessen_und_Versionen.txt
 * (lokal im Verzeichnis ~\.JapaneseVocableTrainer\ des Benutzers bei der Überprüfung auf
 * Neuerungen) bereit.
 *
 * In dieser Datei wird festgehalten, an welchen Gruppen von eigenen Listen der Benutzer Interesse
 * hat und über welche Versionen dieser Listen er bereits informiert wurde.
 *
 * Sie wird bei der Prüfung auf Updates lesend benötigt und schreibend, wenn dem Benutzer der
 * Dialog mit den Neuerungen bei den bereitgestellten eigenen Listen angezeigt wird.
 *
 * Außerdem werden Methoden bereitgestellt, um die Datei zu lesen und aktualisiert wieder zu
 * schreiben.
 *
 * @version 1.01     2024-10-12
 * @author Christian Dühl
 */

public class OwnListVersionAndInterestIo {

    private static final String OWN_LIST_INTERSTS_AND_VERSIONS_BARE_FILENAME =
            "Eigene_Listen_Interessen_und_Versionen.txt";

    /**
     * Der Name der Datei mit dem Interesse des Benutzers und den zuletzt gesehenen Versionen der
     * Gruppen von eigenen Listen, welche zum Download angeboten werden.
     */
    private static final String OWN_LIST_INTERSTS_AND_VERSIONS_FILENAME =
            FileHelper.concatPathes(VOCABLE_TRAINER_DIRECTORY,
                    OWN_LIST_INTERSTS_AND_VERSIONS_BARE_FILENAME);

    /** Lädt die Datei "Eigene_Listen_Interessen_und_Versionen.txt" vom Rechner des Benutzer.  */
    public static List<OwnListInterestAndVersionEntry> loadOwnListVersionAndInterests() {
        String filename = OwnListVersionAndInterestIo.OWN_LIST_INTERSTS_AND_VERSIONS_FILENAME;
        if (!FileHelper.isFile(filename)) {
            return new ArrayList<>();
        }

        String content = FileHelper.readFileToString(filename);
        OwnListVersionAndInterestParser parser = new OwnListVersionAndInterestParser(content);
        parser.parse();
        if (parser.isSuccess()) {
            return parser.getOwnListInterestAndVersionEntries();
        }
        return new ArrayList<>();
    }

    /**
     * Speichert die aktualisierte Datei "Eigene_Listen_Interessen_und_Versionen.txt" auf dem
     * Rechner des Benutzer.
     *
     * @param interestAndVersionEntries
     *            Die Liste der zu speichernden Einträge der Datei.
     */
    public static void storeInterestAndVersionFile(
            List<OwnListInterestAndVersionEntry> interestAndVersionEntries) {
        String filename = OwnListVersionAndInterestIo.OWN_LIST_INTERSTS_AND_VERSIONS_FILENAME;

        List<String> lines = new ArrayList<>();
        for (OwnListInterestAndVersionEntry entry : interestAndVersionEntries) {
            String line = entry.getGroupName()
                    + " - "
                    + BooleanHandling.booleanToString(entry.isInterested())
                    + " - "
                    + entry.getVersion();
            lines.add(line);
        }

        FileHelper.writeLinesToFile(lines, filename);
    }

}
