package de.duehl.basics.io.lock;

/*
 * Copyright 2017 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import de.duehl.basics.io.exceptions.FileNotFoundRuntimeException;
import de.duehl.basics.io.exceptions.NoLockGainedException;

/**
 * Diese Klasse lockt eine Datei und gibt den Lock wieder frei. Wenn der Lock nicht erhalten werden
 * kann, wird eine Ausnahme geworfen.
 *
 * @version 1.01     2017-04-24
 * @author Christian Dühl
 */

public class ThrowingFileLock extends FileLock {

    /**
     * Konstruktor.
     *
     * @param filename
     *            Dateiname
     */
    public ThrowingFileLock(String filename) {
        super(filename);
    }

    /**
     * Konstruktor.
     *
     * @param filename
     *            Dateiname
     * @param rights
     *            Rechte für das Lock:                                                     <ul><li>
     *            "r"   Open for reading only.                                            </li><li>
     *            "rw"  Open for reading and writing.                                     </li><li>
     *            "rws" Open for reading and writing, as with "rw", and
     *                  also require that every update to the file's content
     *                  or metadata be written synchronously to the underlying
     *                  storage device.                                                   </li><li>
     *            "rwd" Open for reading and writing, as with "rw", and also
     *                  require that every update to the file's content be written
     *                  synchronously to the underlying storage device.                  </li></ul>
     */
    public ThrowingFileLock(String filename, String rights) {
        super(filename, rights);
    }

    /**
     * Versucht einen Lock auf die Datei zu erhalten. Der Rückgabewert ist immer true, denn bei
     * Misserfolg wird ja eine Ausnahme geworfen.
     *
     * @return Erfolg
     * @throws FileNotFoundRuntimeException
     *             Wenn die Datei nicht gefunden wurde.
     * @throws NoLockGainedException
     *             Falls das Lock nicht erhalten wurde.
     */
    @Override
    public boolean lock() {
        boolean success = super.lock();
        if (!success) {
            throw new NoLockGainedException("Lock nicht erhalten.");
        }
        return success;
    }

}
