package de.duehl.basics.text.data;

/*
 * Copyright 2023 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import static org.junit.Assert.*;

import java.util.List;

import org.junit.Test;

import de.duehl.basics.collections.CollectionsHelper;

public class FoundSearchTest {

    @Test
    public void notFoundHasIndexMinusOne() {
        FoundSearch foundSearch = FoundSearch.NOT_FOUND;
        int actual = foundSearch.getIndex();
        int expected = -1;
        assertEquals(expected, actual);
    }

    @Test
    public void getIndex() {
        FoundSearch foundSearch = new FoundSearch(42, "Wort");
        int actual = foundSearch.getIndex();
        int expected = 42;
        assertEquals(expected, actual);
    }

    @Test
    public void getSearch() {
        FoundSearch foundSearch = new FoundSearch(42, "Wort");
        String actual = foundSearch.getSearch();
        String expected = "Wort";
        assertEquals(expected, actual);
    }

    @Test
    public void getEnd() {
        FoundSearch foundSearch = new FoundSearch(42, "Wort");
        int actual = foundSearch.getEnd();
        int expected = 46;
        assertEquals(expected, actual);
    }

    @Test
    public void toStringTest() {
        FoundSearch foundSearch = new FoundSearch(42, "Wort");
        String actual = foundSearch.toString();
        String expected = "FoundSearch [index=42, search=Wort]";
        assertEquals(expected, actual);
    }

    @Test
    public void notFoundWasNotSuccessfull() {
        FoundSearch foundSearch = FoundSearch.NOT_FOUND;
        assertFalse(foundSearch.wasSuccessfull());
    }

    @Test
    public void goodFoundWasSuccessfull() {
        FoundSearch foundSearch = new FoundSearch(42, "Wort");
        assertTrue(foundSearch.wasSuccessfull());
    }

    @Test
    public void isBehind() {
        FoundSearch foundSearch1 = new FoundSearch(42, "Wort");
        FoundSearch foundSearch2 = new FoundSearch(57, "anderes Wort");
        assertFalse(foundSearch1.isBehind(foundSearch2));
        assertTrue(foundSearch2.isBehind(foundSearch1));
    }

    @Test
    public void isInAtEnd() {
        FoundSearch shorterfoundSearch = new FoundSearch(65, "Wort");
        FoundSearch longerFoundSearch = new FoundSearch(57, "anderes Wort");
        assertTrue(shorterfoundSearch.isIn(longerFoundSearch));
        assertFalse(longerFoundSearch.isIn(shorterfoundSearch));
    }

    @Test
    public void isInAtBegin() {
        FoundSearch shorterfoundSearch = new FoundSearch(57, "anderes");
        FoundSearch longerFoundSearch = new FoundSearch(57, "anderes Wort");
        assertTrue(shorterfoundSearch.isIn(longerFoundSearch));
        assertFalse(longerFoundSearch.isIn(shorterfoundSearch));
    }

    @Test
    public void isInEqual() {
        FoundSearch shorterfoundSearch = new FoundSearch(57, "anderes Wort");
        FoundSearch longerFoundSearch = new FoundSearch(57, "anderes Wort");
        assertTrue(shorterfoundSearch.isIn(longerFoundSearch));
        assertTrue(longerFoundSearch.isIn(shorterfoundSearch));
    }

    @Test
    public void isInNot() {
        FoundSearch shorterfoundSearch = new FoundSearch(13, "Wort");
        FoundSearch longerFoundSearch = new FoundSearch(57, "anderes Wort");
        assertFalse(shorterfoundSearch.isIn(longerFoundSearch));
        assertFalse(longerFoundSearch.isIn(shorterfoundSearch));
    }

    @Test
    public void sortFoundSearchListByIndex() {
        FoundSearch foundSearch1 = new FoundSearch(42, "Wort");
        FoundSearch foundSearch2 = new FoundSearch(57, "anderes Wort");
        FoundSearch foundSearch3 = new FoundSearch(48, "ha");
        List<FoundSearch> list = CollectionsHelper.buildListFrom(foundSearch1, foundSearch2,
                foundSearch3);
        assertEquals(3, list.size());
        assertEquals("Wort", list.get(0).getSearch());
        assertEquals(42, list.get(0).getIndex());
        assertEquals("anderes Wort", list.get(1).getSearch());
        assertEquals(57, list.get(1).getIndex());
        assertEquals("ha", list.get(2).getSearch());
        assertEquals(48, list.get(2).getIndex());

        FoundSearch.sortFoundSearchListByIndex(list);
        assertEquals(3, list.size());
        assertEquals("Wort", list.get(0).getSearch());
        assertEquals(42, list.get(0).getIndex());
        assertEquals("ha", list.get(1).getSearch());
        assertEquals(48, list.get(1).getIndex());
        assertEquals("anderes Wort", list.get(2).getSearch());
        assertEquals(57, list.get(2).getIndex());
    }

    @Test
    public void sortFoundSearchListReverseByEndAndIndex() {
        FoundSearch foundSearch1 = new FoundSearch(42, "Wort");
        FoundSearch foundSearch2 = new FoundSearch(34, "anderes Wort");
        FoundSearch foundSearch3 = new FoundSearch(22, "ha");
        List<FoundSearch> list = CollectionsHelper.buildListFrom(foundSearch1, foundSearch2,
                foundSearch3);
        assertEquals(3, list.size());
        assertEquals("Wort", list.get(0).getSearch());
        assertEquals(42, list.get(0).getIndex());
        assertEquals("anderes Wort", list.get(1).getSearch());
        assertEquals(34, list.get(1).getIndex());
        assertEquals("ha", list.get(2).getSearch());
        assertEquals(22, list.get(2).getIndex());

        FoundSearch.sortFoundSearchListReverseByEndAndIndex(list);
        assertEquals(3, list.size());
        assertEquals("anderes Wort", list.get(0).getSearch());
        assertEquals(34, list.get(0).getIndex());
        assertEquals("Wort", list.get(1).getSearch());
        assertEquals(42, list.get(1).getIndex());
        assertEquals("ha", list.get(2).getSearch());
        assertEquals(22, list.get(2).getIndex());
    }

}
