package de.duehl.swing.persistance;

/*
 * Copyright 2024 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Point;
import java.util.ArrayList;
import java.util.List;

import de.duehl.basics.text.NumberString;
import de.duehl.basics.text.Text;

/**
 * Diese Klasse regelt den Umgang mit Informationen, die zwischen zwei Läufen des Programms
 * bewahrt werden sollen. Sie stellt Methoden bereit, diese Daten zu speichern und zu lesen.
 *
 * @version 1.01     2024-07-26
 * @author Christian Dühl
 */

public class SwingSessionManagerHelper {

    public static final Point NOT_INITIALISED_POINT = new Point(-1, -1);
    public static final Dimension NOT_INITIALISED_DIMENSION = new Dimension(-1, -1);


    /** Erzeugt aus einem Punkt einen String, um diesen persistent abzuspeichern. */
    public static String createPointStorageString(Point point) {
        return Integer.toString(point.x) + "#" + Integer.toString(point.y);
    }

    /** Erzeugt aus einem persistent abgespeicherten String wieder einen Punkt. */
    public static Point createPointFromStorageString(String storageString) {
        List<String> parts = Text.splitBy(storageString, "#");
        if (parts.size() != 2) {
            return NOT_INITIALISED_POINT;
        }
        else {
            String part1 = parts.get(0);
            String part2 = parts.get(1);
            if (NumberString.isDigitSequence(part1) && NumberString.isDigitSequence(part2)) {
                int x = NumberString.parseInt(part1);
                int y = NumberString.parseInt(part2);
                return new Point(x, y);
            }
            else {
                return NOT_INITIALISED_POINT;
            }
        }
    }

    /** Erzeugt aus einer Dimension einen String, um diesen persistent abzuspeichern. */
    public static String createDimensionStorageString(Dimension dimension) {
        return Integer.toString(dimension.width) + "#" + Integer.toString(dimension.height);
    }

    /** Erzeugt aus einem persistent abgespeicherten String wieder eine Dimension. */
    public static Dimension createDimensionFromStorageString(String storageString) {
        List<String> parts = Text.splitBy(storageString, "#");
        if (parts.size() != 2) {
            return NOT_INITIALISED_DIMENSION;
        }
        else {
            String part1 = parts.get(0);
            String part2 = parts.get(1);
            if (NumberString.isDigitSequence(part1) && NumberString.isDigitSequence(part2)) {
                int width = NumberString.parseInt(part1);
                int height = NumberString.parseInt(part2);
                return new Dimension(width, height);
            }
            else {
                return NOT_INITIALISED_DIMENSION;
            }
        }
    }

    /** Erzeugt aus einem Farbe einen String, um diesen persistent abzuspeichern. */
    public static String createColorStorageString(Color color) {
        return Integer.toString(color.getRed())
                + "#" + Integer.toString(color.getGreen())
                + "#" + Integer.toString(color.getBlue())
                + "#" + Integer.toString(color.getAlpha())
                ;
    }

    /**
     * Erzeugt aus einem persistent abgespeicherten String wieder einen Farbe.
     *
     * Da man leider keine Color erzeugen kann mit Werten, die außerhalb des Bereichs 0-255 liegen,
     * kann man leider keine NOT_INITIALISED_COLOR definieren. Geht etwas schief, wird statt dessen
     * Weiß zurückgegeben.
     */
    public static Color createColorFromStorageString(String storageString) {
        List<String> parts = Text.splitBy(storageString, "#");
        if (parts.size() != 4) {
            return Color.WHITE;
        }
        else {
            String part1 = parts.get(0);
            String part2 = parts.get(1);
            String part3 = parts.get(2);
            String part4 = parts.get(3);
            if (NumberString.isDigitSequence(part1)
                    && NumberString.isDigitSequence(part2)
                    && NumberString.isDigitSequence(part3)
                    && NumberString.isDigitSequence(part4)
                    ) {
                int red = NumberString.parseInt(part1);
                int green = NumberString.parseInt(part2);
                int blue = NumberString.parseInt(part3);
                int alpha = NumberString.parseInt(part4);
                return new Color(red, green, blue, alpha);
            }
            else {
                return Color.WHITE;
            }
        }
    }

    private static final String COLORS_SEPARATOR = "#-#";

    /** Erzeugt aus einer Liste von Farben einen String, um diesen persistent abzuspeichern. */
    public static String createColorsStorageString(List<Color> colors) {
        List<String> storageStrings = new ArrayList<>();
        for (Color color : colors) {
            String storageString = createColorStorageString(color);
            storageStrings.add(storageString);
        }
        return Text.join(COLORS_SEPARATOR, storageStrings);
    }

    /**
     * Erzeugt aus einem persistent abgespeicherten String wieder eine Liste von Farben.
     *
     * Da man leider keine Color erzeugen kann mit Werten, die außerhalb des Bereichs 0-255 liegen,
     * kann man leider keine NOT_INITIALISED_COLOR definieren. Geht etwas schief, wird statt dessen
     * jeweils Weiß zurückgegeben.
     */
    public static List<Color> createColorsListFromStorageString(String listStorageString) {
        List<String> storageStrings = Text.splitBy(listStorageString, COLORS_SEPARATOR);

        List<Color> colors = new ArrayList<>();
        for (String storageString : storageStrings) {
            Color color = createColorFromStorageString(storageString);
            colors.add(color);
        }

        return colors;
    }

}
