package de.duehl.swing.ui.buttons;

/*
 * Copyright 2024 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.BorderLayout;
import java.awt.Insets;
import java.awt.image.BufferedImage;

import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JPanel;

import de.duehl.swing.ui.GuiTools;

/**
 * Diese Klasse stellt ein Button-Paar dar, welches eine Aktion über einen bebilderten Button
 * (optional) oder über einen Text-Button (darunter) auslöst.
 *
 * @version 1.01     2024-06-27
 * @author Christian Dühl
 */

public class ThumbnailAndTextButtonPair {

    private static final int DEFAULT_MAX_TEXT_LENGTH = 50;


    /** Der Text für den Text-Button. */
    private final String text;

    /** Der Panel auf dem das Element erstellt wird. */
    private final JPanel panel;

    private boolean thumbnailShown = false;

    /** Die maximale Länge der Texte in den Buttons, nach dieser wird der Text umgebrochen. */
    private int maxTextLenght;

    /* --- andere Gui-Elemente --- */
    private final JButton thumbnailButton;
    private final JButton textButton;

    /**
     * Konstruktor.
     *
     * @param text
     *            Der Text für den Text-Button.
     */
    public ThumbnailAndTextButtonPair(String text) {
        this.text = text;

        thumbnailShown = false;
        maxTextLenght = DEFAULT_MAX_TEXT_LENGTH;

        panel = new JPanel();
        thumbnailButton = new JButton();
        textButton = new JButton();

        initElements();
    }

    /** Legt das anzuzeigende Miniaturbild fest. */
    public void setThumbnail(BufferedImage thumbnail) {
        thumbnailShown = true;
        thumbnailButton.setIcon(new ImageIcon(thumbnail));
        thumbnailButton.setMargin(new Insets(0, 0, 5, 0)); // Abstand zum Text
    }

    /**
     * Setter für die maximale Länge der Texte in den Buttons, nach dieser wird der Text
     * umgebrochen.
     */
    public void setMaxTextLenght(int maxTextLenght) {
        this.maxTextLenght = maxTextLenght;
    }

    private void initElements() {
        initTextButton();
        dispatchMouseScrollEvent();
    }

    private void initTextButton() {
        String collectionText = createCollectionText();
        textButton.setText(collectionText);
    }

    private void dispatchMouseScrollEvent() {
        GuiTools.dispatchMouseScrollEvent(panel, 4);
        GuiTools.dispatchMouseScrollEvent(thumbnailButton, 5);
        GuiTools.dispatchMouseScrollEvent(textButton, 5);
    }

    private String createCollectionText() {
        return GuiTools.createHtmlTextForButtonsWithUnderscoreReplacement(text, maxTextLenght);
    }

    /** Erstellt die Oberfläche mit den Buttons nach den vorangegangenen Einstellungen. */
    public void createGui() {
        panel.setLayout(new BorderLayout());

        if (thumbnailShown) {
            panel.add(thumbnailButton, BorderLayout.CENTER);
        }
        panel.add(textButton, BorderLayout.SOUTH);
    }

    /** Fügt eine Aktion hinzu, die auf Auslösen eines der beiden Buttons hin ausgelöst wird. */
    public void addActionListener(Runnable runabble) {
        textButton.addActionListener(e -> runabble.run());
        thumbnailButton.addActionListener(e -> runabble.run());
    }

    /** Getter für den Panel auf dem das Element erstellt wird. */
    public JPanel getPanel() {
        return panel;
    }

}
