package de.duehl.swing.ui.menu;

/*
 * Copyright 2020 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.event.ActionListener;

import javax.swing.JCheckBoxMenuItem;
import javax.swing.KeyStroke;

import de.duehl.swing.ui.menu.collection.AutomaticMyMenuItemCollection;
import de.duehl.swing.ui.menu.collection.StoredMyMenuItem;

/**
 * Diese Klasse stellt eine kleine Erweiterung des JMenuItems dar, um JMenuItems hübscher erzeugen
 * zu können.
 *
 * Verwendung mit:
 *
 *     return new MyCheckBoxMenuItem("Aus einem Verzeichnis laden ...")
 *             .mnemonic('l')
 *             .accelerator(KeyEvent.VK_L, InputEvent.CTRL_DOWN_MASK)
 *             .actionListener(() -> logic.loadFromDisk())
 *             ;
 *
 * @version 1.01     2020-07-08
 * @author Christian Dühl
 */

public class MyCheckBoxMenuItem extends JCheckBoxMenuItem {

    private static final long serialVersionUID = 1L;

    /** Ein zugehöriges Objekt, das sich die vergebenen Eigenschaften merkt. */
    private final StoredMyMenuItem storedMyMenuItem;

    /**
     * Konstruktor.
     *
     * @param text
     *            Dargestellter Text des Menüeintrags.
     */
    public MyCheckBoxMenuItem(String text) {
        super(text);
        storedMyMenuItem = new StoredMyMenuItem(text);
    }

    /**
     * Legt das Zeichen des Menüeintrags, das unterstrichen wird, auf den übergebenen char-Wert
     * fest.
     */
    public MyCheckBoxMenuItem mnemonic(char mnemonic) {
        storedMyMenuItem.mnemonic(mnemonic);
        setMnemonic(mnemonic);
        return this;
    }

    /**
     * Legt das Zeichen des Menüeintrags, das unterstrichen wird, auf den übergebenen int-Wert fest
     * (wird auf char gecastet).
     */
    public MyCheckBoxMenuItem mnemonic(int mnemonic) {
        return mnemonic((char) mnemonic);
    }

    /**
     * Legt das Tasten-Kürzel des Menüeintrags fest.
     *
     * @param keyCode
     *            Der numerische Wert einer Taste auf der Tastatur für das Tasten-Kürzel des
     *            Menüeintrags.
     * @param modifiers
     *            Bitweise ODER-verknüpfte Modifikatoren für das Tasten-Kürzel des Menüeintrags.
     */
    public MyCheckBoxMenuItem accelerator(int keyCode, int modifiers) {
        storedMyMenuItem.accelerator(keyCode, modifiers);
        setAccelerator(KeyStroke.getKeyStroke(keyCode, modifiers));
        return this;
    }

    /** Legt Aktion fest, die ausgeführt wird, wenn der Menüeintrag angeklickt wird. */
    public MyCheckBoxMenuItem actionListener(ActionListener actionListener) {
        storedMyMenuItem.actionListener(actionListener);
        addActionListener(actionListener);
        return this;
    }

    /** Schaltet die automatische Sammlung von Menüeinträgen ab. */
    public static void switchOffMyMenuItemCollection() {
        AutomaticMyMenuItemCollection.switchOffMyMenuItemCollection();
    }

    /**
     * Schaltet die automatische Sammlung von Menüeinträgen an (nur sinnvoll, falls vorher selbst
     * abgeschaltet).
     */
    public static void switchOnMyMenuItemCollection() {
        AutomaticMyMenuItemCollection.switchOnMyMenuItemCollection();
    }

}
