package de.duehl.swing.ui.text.html;

/*
 * Copyright 2020 Christian Dühl. All rights reserved.
 *
 * This program is free software. You can redistribute it and/or
 * modify it under the same terms as perl:
 *
 * general:  http://dev.perl.org/licenses/
 * GPL:      http://dev.perl.org/licenses/gpl1.html
 * artistic: http://dev.perl.org/licenses/artistic.html
 */

import java.awt.BorderLayout;

import javax.swing.JButton;

import java.awt.Image;
import java.awt.Point;

import de.duehl.swing.ui.dialogs.base.ModalDialogBase;

/**
 * Diese Klasse öffnet einen Dialog um einen Text in HTML in einem Textfeld anzuzeigen und den
 * Benutzer mit Ja oder Nein antworten zu lassen.
 *
 * @version 1.01     2020-03-04
 * @author Christian Dühl
 */

public final class HtmlYesNoDialog extends ModalDialogBase {

    /** Der eingebettete Panel mit HTML, Quellcode und Buttons. */
    private final HtmlAndSourceCodePanel htmlPanel;

    /** Antwort des Benutzers. */
    private boolean answer;

    private JButton noButton;
    private JButton yesButton;

    /**
     * Konstruktor.
     *
     * @param title
     *            Titel des Dialogs.
     */
    public HtmlYesNoDialog(String title) {
        this(title, new Point(100, 100));
    }

    /**
     * Konstruktor.
     *
     * @param title
     *            Titel des Dialogs.
     * @param parentLocation
     *            Lokation des Aufrufenden Dialoges.
     */
    public HtmlYesNoDialog(String title, Point parentLocation) {
        this(title, null, parentLocation);
    }

    /**
     * Konstruktor.
     *
     * @param title
     *            Titel des Dialogs.
     * @param programImage
     *            Image vom aufrufenden Programm.
     * @param parentLocation
     *            Lokation des Aufrufenden Dialoges.
     */
    public HtmlYesNoDialog(String title, Image programImage, Point parentLocation) {
        super(parentLocation, programImage, title);
        addEscapeBehaviour();

        answer = false;
        noButton = createNoButton();
        yesButton = createYesButton();

        htmlPanel = HtmlAndSourceCodePanel.create();
        htmlPanel.setKeyBindings(getRootPane());
        htmlPanel.addButtonOnLeft(noButton);
        htmlPanel.addButtonOnRight(yesButton);

        fillDialog();
    }

    private JButton createNoButton() {
        JButton button = new JButton("Nein");
        button.addActionListener(e -> no());
        return button;
    }

    private JButton createYesButton() {
        JButton button = new JButton("Ja");
        button.addActionListener(e -> yes());
        return button;
    }

    private void no() {
        answer = false;
        closeDialog();
    }

    private void yes() {
        answer = true;
        closeDialog();
    }

    @Override
    protected void populateDialog() {
        add(htmlPanel.getComponent(), BorderLayout.CENTER);
    }

//    /** Fügt einen weiteren Button rechts unten zum Dialog hinzu. */
//    public void addButtonOnRight(Component button) {
//        htmlPanel.addButtonOnRight(button);
//    }
//
//    /** Fügt einen weiteren Button links unten zum Dialog hinzu. */
//    public void addButtonOnLeft(Component button) {
//        htmlPanel.addButtonOnLeft(button);
//    }
//
//    /**
//     * Schreibt den Text der Seite hinter der URL in den Hauptteil des Dialogs und setzt den
//     * Textcursor auf den Anfang des Dokuments.
//     *
//     * @param url
//     *            URL, aus der der Inhalt gelesen wird.
//     */
//    public void showHtml(URL url) {
//        htmlPanel.showHtml(url);
//    }

    /**
     * Schreibt das HTML in den Anzeigebereich und setzt den Textcursor auf den Anfang des
     * Dokuments.
     *
     * @param html
     *            Darzustellender Text.
     */
    public void setText(String html) {
        htmlPanel.setText(html);
        htmlPanel.scrollScrollbarToMinimumLater();
    }

//    /** Scrollt zum Anfang. */
//    public void scrollScrollbarToMinimumLater() {
//        htmlPanel.scrollScrollbarToMinimumLater();
//    }
//
//    /** Scrollt zum Ende. */
//    public void scrollScrollbarToMaximumLater() {
//        htmlPanel.scrollScrollbarToMaximumLater();
//    }
//
//    /** Scrollt weiter nach oben. */
//    public void scrollScrollbarToPreviousSectionLater() {
//        htmlPanel.scrollScrollbarToPreviousSectionLater();
//    }
//
//    /** Scrollt weiter nach unten. */
//    public void scrollScrollbarToNextSectionLater() {
//        htmlPanel.scrollScrollbarToNextSectionLater();
//    }

    /** Getter für die Antwort des Benutzers. */
    public boolean getUserAnswer() {
        return answer;
    }

}
