package de.duehl.vocabulary.japanese.startup.logic.steps;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import de.duehl.basics.collections.CollectionsHelper;
import de.duehl.basics.datetime.time.watch.StopWatch;
import de.duehl.basics.text.Text;
import de.duehl.vocabulary.japanese.common.persistence.Options;
import de.duehl.vocabulary.japanese.data.KanjiSet;
import de.duehl.vocabulary.japanese.data.symbol.Kanji;
import de.duehl.vocabulary.japanese.io.AllKanjiSetsReader;
import de.duehl.vocabulary.japanese.startup.ui.data.SplashScreenable;

/**
 * Diese Klasse steht für den Schritt zum Einlesen Kanji-Mengen beim Startup des Vokabeltrainers.
 *
 * @version 1.01     2025-11-24
 * @author Christian Dühl
 */

public class Step13LoadKanjiSets extends StartupStep {

    /** Die Liste der benutzerdefinierten Kanji-Mengen. */
    private List<KanjiSet> kanjiSets;

    /**
     * Konstruktor.
     *
     * @param step
     *            Der Schritt der durchgeführt wird.
     * @param options
     *            Die Programmoptionen.
     * @param splashScreen
     *            Die grafische Oberfläche beim Start in der die Meldungen angezeigt werden.
     * @param watch
     *            Misst die Laufzeit des gesamten Startups.
     */
    public Step13LoadKanjiSets(String step, Options options, SplashScreenable splashScreen,
            StopWatch watch) {
        super(step, options, splashScreen, watch);
    }

    /** Führt den eigentlichen Inhalt des Schritts aus. */
    @Override
    protected void runInternalStep() {
        loadKanjiSets();

        // Durch den Editor kann eigentlich kein Kanji mehrfach in der gleichen Kanji-Menge sein.
        // Daher prüfe ich das hier nicht.

        checkIfKanjiAreInMultipleKanjiSetsOfSameGroup();
    }

    private void loadKanjiSets() {
        appendMessage("Lade die benutzerdefinierten Kanji-Mengen ...");
        kanjiSets = AllKanjiSetsReader.read();
    }

    private void checkIfKanjiAreInMultipleKanjiSetsOfSameGroup() {
        Map<String, List<KanjiSet>> kanjiSetsByGroup = createKanjiSetsByGroup();

        for (String group : CollectionsHelper.getSortedMapStringIndices(kanjiSetsByGroup)) {
            List<KanjiSet> groupList = kanjiSetsByGroup.get(group);
            checkIfKanjiAreInMultipleKanjiSetsOfSameGroup(groupList);
        }
    }

    private Map<String, List<KanjiSet>> createKanjiSetsByGroup() {
        Map<String, List<KanjiSet>> kanjiSetsByGroup = new HashMap<>();

        for (KanjiSet kanjiSet : kanjiSets) {
            String group = kanjiSet.getGroup();
            if (!kanjiSetsByGroup.containsKey(group)) {
                kanjiSetsByGroup.put(group, new ArrayList<>());
            }
            List<KanjiSet> groupList = kanjiSetsByGroup.get(group);
            groupList.add(kanjiSet);
        }

        return kanjiSetsByGroup;
    }

    private void checkIfKanjiAreInMultipleKanjiSetsOfSameGroup(List<KanjiSet> groupList) {
        Map<Kanji, List<String>> namesByKanji = createNamesByKanji(groupList);

        for (Kanji kanji : namesByKanji.keySet()) {
            List<String> namesList = namesByKanji.get(kanji);
            if (namesList.size() > 1) {
                List<String> warnings = new ArrayList<>();
                warnings.add("Das Kanji 「" + kanji.getCharacter() + "」 ist in mehr als einer "
                        + "Kanji-Menge der gleichen Gruppe enthalten:");
                for (String name : namesList) {
                    warnings.add("    in Kanji-Menge '" + name + "'");
                }
                String warningsString = Text.joinWithLineBreak(warnings);
                appendMessage(warningsString);
                warningsInStep(warningsString);
            }
        }
    }

    private Map<Kanji, List<String>> createNamesByKanji(List<KanjiSet> groupList) {
        Map<Kanji, List<String>> namesByKanji = new HashMap<>();

        for (KanjiSet kanjiSet : groupList) {
            for (Kanji kanji : kanjiSet.getSet()) {
                if (!namesByKanji.containsKey(kanji)) {
                    namesByKanji.put(kanji, new ArrayList<>());
                }
                List<String> namesList = namesByKanji.get(kanji);
                namesList.add(kanjiSet.getName());
            }
        }

        return namesByKanji;
    }

    /** Getter für die Liste der benutzerdefinierten Kanji-Mengen. */
    public List<KanjiSet> getKanjiSets() {
        return kanjiSets;
    }

}
