package de.duehl.vocabulary.japanese.ui.dialog.grammar.headers.io;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import de.duehl.basics.collections.CollectionsHelper;
import de.duehl.basics.io.Charset;
import de.duehl.basics.io.FileHelper;
import de.duehl.basics.text.Text;
import de.duehl.swing.ui.GuiTools;
import de.duehl.vocabulary.japanese.common.persistence.GrammarTableHeaderPersistenceHelper;
import de.duehl.vocabulary.japanese.ui.dialog.grammar.headers.data.GrammarFormAndStringSelection;
import de.duehl.vocabulary.japanese.ui.dialog.grammar.headers.tools.GrammarTableHeaderFieldsToMapCreator;

/**
 * Diese Klasse speichert aus dem Dialoge zur Bearbeitung der Überschriften für die Erstellung von
 * grammatikalischen Tabellen von Verben oder Adjektiven die vom Benutzer eingegebenen
 * Überschriften in einer Datei, damit zwischen verschiedenen Versionen dieser Überschriften
 * gewechselt werden kann.
 *
 * @version 1.01     2025-11-29
 * @author Christian Dühl
 */

public class GrammarTableOwnHeadersSaver {

    /** Der Dateiname. */
    private final String filename;

    /** Die Checkboxen für die Verbformen. */
    private final List<GrammarFormAndStringSelection> verbGrammarFormAndStringSelections;

    /** Die Checkboxen für die I-Adjektivformen. */
    private final List<GrammarFormAndStringSelection> iAdjectiveGrammarFormAndStringSelections;

    /** Die Checkboxen für die NA-Adjektivformen. */
    private final List<GrammarFormAndStringSelection> naAdjectiveGrammarFormAndStringSelections;

    /** Gibt an, ob das Auslesen der Felder erfolgreich war. */
    private boolean success;

    /** Die während des Auslesen aufgetretenen Probleme. */
    private List<String> problems;

    /**
     * Konstruktor.
     *
     * @param filename
     *            Der Dateiname.
     * @param verbGrammarFormAndStringSelections
     *            Die Checkboxen für die Verbformen.
     * @param iAdjectiveGrammarFormAndStringSelections
     *            Die Checkboxen für die I-Adjektivformen.
     * @param naAdjectiveGrammarFormAndStringSelections
     *            Die Checkboxen für die NA-Adjektivformen.
     */
    public GrammarTableOwnHeadersSaver(String filename,
            List<GrammarFormAndStringSelection> verbGrammarFormAndStringSelections,
            List<GrammarFormAndStringSelection> iAdjectiveGrammarFormAndStringSelections,
            List<GrammarFormAndStringSelection> naAdjectiveGrammarFormAndStringSelections) {
        this.filename = filename;
        this.verbGrammarFormAndStringSelections = verbGrammarFormAndStringSelections;
        this.iAdjectiveGrammarFormAndStringSelections =
                iAdjectiveGrammarFormAndStringSelections;
        this.naAdjectiveGrammarFormAndStringSelections =
                naAdjectiveGrammarFormAndStringSelections;
    }

    /** Speichert die Datei. */
    public void save() {
        Map<String, String> verbHeaderMap = new HashMap<>();
        Map<String, String> iAdjectiveHeaderMap = new HashMap<>();
        Map<String, String> naAdjectiveHeaderMap = new HashMap<>();

        success = true;
        problems = new ArrayList<>();

        storeHeadersFromSelectionsIntoMap(verbGrammarFormAndStringSelections,
                verbHeaderMap);
        storeHeadersFromSelectionsIntoMap(iAdjectiveGrammarFormAndStringSelections,
                iAdjectiveHeaderMap);
        storeHeadersFromSelectionsIntoMap(naAdjectiveGrammarFormAndStringSelections,
                naAdjectiveHeaderMap);

        if (success) {
            String verbLine = GrammarTableHeaderPersistenceHelper
                    .createGrammarTableHeaderPesistenceString(verbHeaderMap);
            String iAdjectiveLine = GrammarTableHeaderPersistenceHelper
                    .createGrammarTableHeaderPesistenceString(iAdjectiveHeaderMap);
            String naAdjectiveLine = GrammarTableHeaderPersistenceHelper
                    .createGrammarTableHeaderPesistenceString(naAdjectiveHeaderMap);
            List<String> lines = CollectionsHelper.buildListFrom(verbLine, iAdjectiveLine,
                    naAdjectiveLine);
            FileHelper.writeLinesToFile(lines, filename, Charset.UTF_8);
        }
        else {
            GuiTools.informUser(
                    "Die Überschriften können nicht gespeichert werden",
                    Text.join("\n\n", problems));
        }
    }

    private void storeHeadersFromSelectionsIntoMap(
            List<GrammarFormAndStringSelection> formAndSelections, Map<String, String> headerMap) {
        GrammarTableHeaderFieldsToMapCreator creator =
                new GrammarTableHeaderFieldsToMapCreator(formAndSelections, headerMap);
        creator.create();
        if (!creator.isSuccess()) {
            success = false;
            problems.addAll(creator.getProblems());
        }
    }

}
